<?php
/**
 * Plugin Name: PayBridge Payment Gateway for WooCommerce
 * Plugin URI: https://paybridge.my
 * Description: Accept payments via PayBridge payment gateway (FPX, Credit Card, E-Wallet) for WooCommerce
 * Version: 1.0.0
 * Author: PayBridge
 * Author URI: https://paybridge.my
 * License: GPL-2.0+
 * Text Domain: paybridge-woocommerce
 * WC requires at least: 3.0
 * WC tested up to: 8.0
 */

if (!defined('ABSPATH')) {
    exit;
}

// Check if WooCommerce is active
if (!in_array('woocommerce/woocommerce.php', apply_filters('active_plugins', get_option('active_plugins')))) {
    return;
}

// Add the gateway to WooCommerce
add_filter('woocommerce_payment_gateways', 'paybridge_add_gateway');
function paybridge_add_gateway($gateways) {
    $gateways[] = 'WC_PayBridge_Gateway';
    return $gateways;
}

// Initialize the gateway class
add_action('plugins_loaded', 'paybridge_init_gateway');
function paybridge_init_gateway() {

    class WC_PayBridge_Gateway extends WC_Payment_Gateway {

        public function __construct() {
            $this->id = 'paybridge';
            $this->icon = apply_filters('woocommerce_paybridge_icon', plugins_url('assets/paybridge-logo.png', __FILE__));
            $this->has_fields = false;
            $this->method_title = 'PayBridge';
            $this->method_description = 'Accept payments via PayBridge (FPX, Credit Card, E-Wallet)';

            // Load settings
            $this->init_form_fields();
            $this->init_settings();

            // Define user set variables
            $this->title = $this->get_option('title');
            $this->description = $this->get_option('description');
            $this->client_id = $this->get_option('client_id');
            $this->client_secret = $this->get_option('client_secret');
            $this->api_url = $this->get_option('api_url');
            $this->sandbox = $this->get_option('sandbox');

            // Actions
            add_action('woocommerce_update_options_payment_gateways_' . $this->id, array($this, 'process_admin_options'));
            add_action('woocommerce_api_paybridge_callback', array($this, 'handle_callback'));
            add_action('woocommerce_api_paybridge_return', array($this, 'handle_return'));
        }

        public function init_form_fields() {
            $this->form_fields = array(
                'enabled' => array(
                    'title' => 'Enable/Disable',
                    'type' => 'checkbox',
                    'label' => 'Enable PayBridge Payment',
                    'default' => 'yes'
                ),
                'title' => array(
                    'title' => 'Title',
                    'type' => 'text',
                    'description' => 'Payment method title that customers see during checkout.',
                    'default' => 'PayBridge (FPX/Card/E-Wallet)',
                    'desc_tip' => true,
                ),
                'description' => array(
                    'title' => 'Description',
                    'type' => 'textarea',
                    'description' => 'Payment method description that customers see during checkout.',
                    'default' => 'Pay securely via FPX, Credit Card, or E-Wallet through PayBridge.',
                ),
                'api_url' => array(
                    'title' => 'PayBridge API URL',
                    'type' => 'text',
                    'description' => 'Your PayBridge installation URL (e.g., https://paybridge.my)',
                    'default' => 'https://paybridge.my',
                ),
                'client_id' => array(
                    'title' => 'Client ID',
                    'type' => 'text',
                    'description' => 'Your PayBridge Merchant Client ID',
                ),
                'client_secret' => array(
                    'title' => 'Client Secret',
                    'type' => 'password',
                    'description' => 'Your PayBridge Merchant Client Secret',
                ),
                'sandbox' => array(
                    'title' => 'Sandbox Mode',
                    'type' => 'checkbox',
                    'label' => 'Enable Sandbox/Test Mode',
                    'default' => 'yes',
                    'description' => 'Use sandbox mode for testing.',
                ),
            );
        }

        public function process_payment($order_id) {
            $order = wc_get_order($order_id);

            // Get access token
            $token = $this->get_access_token();

            if (!$token) {
                wc_add_notice('Payment error: Unable to connect to payment gateway.', 'error');
                return;
            }

            // Create transaction
            $transaction = $this->create_transaction($order, $token);

            if (!$transaction || $transaction['status'] != 'success') {
                wc_add_notice('Payment error: ' . ($transaction['message'] ?? 'Unable to create transaction.'), 'error');
                return;
            }

            // Redirect to PayBridge payment page
            return array(
                'result' => 'success',
                'redirect' => $transaction['data']['approvalUrl']
            );
        }

        private function get_access_token() {
            $url = rtrim($this->api_url, '/') . '/merchant/api/verify';

            $response = wp_remote_post($url, array(
                'timeout' => 30,
                'body' => array(
                    'client_id' => $this->client_id,
                    'client_secret' => $this->client_secret,
                ),
            ));

            if (is_wp_error($response)) {
                return false;
            }

            $body = json_decode(wp_remote_retrieve_body($response), true);

            if ($body['status'] == 'success') {
                return $body['data']['accessToken'];
            }

            return false;
        }

        private function create_transaction($order, $token) {
            $url = rtrim($this->api_url, '/') . '/merchant/api/transaction-info';

            $callback_url = add_query_arg('wc-api', 'paybridge_callback', home_url('/'));
            $return_url = add_query_arg(array(
                'wc-api' => 'paybridge_return',
                'order_id' => $order->get_id(),
                'order_key' => $order->get_order_key(),
            ), home_url('/'));

            $response = wp_remote_post($url, array(
                'timeout' => 30,
                'headers' => array(
                    'Authorization' => 'Bearer ' . $token,
                    'Content-Type' => 'application/json',
                ),
                'body' => json_encode(array(
                    'payer' => 'Mts',
                    'amount' => $order->get_total(),
                    'currency' => $order->get_currency(),
                    'successUrl' => $return_url,
                    'cancelUrl' => $order->get_cancel_order_url(),
                    'order_id' => $order->get_id(),
                    'item_name' => 'Order #' . $order->get_order_number(),
                    'payer_name' => $order->get_billing_first_name() . ' ' . $order->get_billing_last_name(),
                    'payer_email' => $order->get_billing_email(),
                    'payer_phone' => $order->get_billing_phone(),
                )),
            ));

            if (is_wp_error($response)) {
                return false;
            }

            return json_decode(wp_remote_retrieve_body($response), true);
        }

        public function handle_callback() {
            // Handle server-to-server callback from PayBridge
            $status = isset($_REQUEST['status']) ? sanitize_text_field($_REQUEST['status']) : '';
            $order_id = isset($_REQUEST['order_id']) ? intval($_REQUEST['order_id']) : 0;
            $transaction_id = isset($_REQUEST['transaction_id']) ? sanitize_text_field($_REQUEST['transaction_id']) : '';

            if ($order_id) {
                $order = wc_get_order($order_id);

                if ($order) {
                    if ($status == 'success' || $status == '1') {
                        $order->payment_complete($transaction_id);
                        $order->add_order_note('Payment completed via PayBridge. Transaction ID: ' . $transaction_id);
                    } else {
                        $order->update_status('failed', 'Payment failed via PayBridge.');
                    }
                }
            }

            echo 'OK';
            exit;
        }

        public function handle_return() {
            // Handle customer return from PayBridge
            $order_id = isset($_REQUEST['order_id']) ? intval($_REQUEST['order_id']) : 0;
            $order_key = isset($_REQUEST['order_key']) ? sanitize_text_field($_REQUEST['order_key']) : '';
            $status = isset($_REQUEST['status']) ? sanitize_text_field($_REQUEST['status']) : '';

            if ($order_id && $order_key) {
                $order = wc_get_order($order_id);

                if ($order && $order->get_order_key() === $order_key) {
                    if ($status == 'success' || $status == '1' || $order->is_paid()) {
                        wp_redirect($this->get_return_url($order));
                        exit;
                    } else {
                        wc_add_notice('Payment was not completed. Please try again.', 'error');
                        wp_redirect(wc_get_checkout_url());
                        exit;
                    }
                }
            }

            wp_redirect(home_url());
            exit;
        }
    }
}

// Add settings link on plugin page
add_filter('plugin_action_links_' . plugin_basename(__FILE__), 'paybridge_settings_link');
function paybridge_settings_link($links) {
    $settings_link = '<a href="admin.php?page=wc-settings&tab=checkout&section=paybridge">Settings</a>';
    array_unshift($links, $settings_link);
    return $links;
}
